/**
 * lexer.jflex - A lexical analyser for RDF. 
 *
 *  (full unicode version)
 * 
 * @author Karsten Tolle
 * @version 2.5  (30 Januar 2003) */

// Part one: user code

package gr.forth.ics.vrp.corevrp.parser;

import java_cup.runtime.*; // convenience, but not necessary
import java.util.*; // for the Stack
import gr.forth.ics.vrp.corevrp.*;
import gr.forth.ics.vrp.corevrp.vocabulary.*;

%%

// Part two: options and declarations

%class Lexer	// name of the created class file
%public
%unicode	// using unicode 16 bit
%cup		// work together with CUP
%line		// give line-number during parsing 
%column		// give column-number during parsing 
%pack

		 // added 01. August 2001 by Karsten Tolle 
		 // new since JFlex 1.3 here we increase the default buffer size
%buffer 32768

%{
   /**
    * NS_Prefix is for saving the prefixes of the RDF namespace.
    */
   private String NS_Prefix;

   /**
    * RDF_prefix_set containing the prefixes for the RDF namespace.
    */
   private HashSet RDF_prefix_set = new HashSet();

   /**
    * Boolean set if we have a namespace declaration.
    */
    private boolean ns = false;

   /**
    * The nested_counter helps to know when we are allowed to leave a RDF description
    * in case we have a embedded RDF.
    */
   private int nested_counter = 0;

   /**
    * Hash Map containing the entity reference names and values.
    * new since 14. Jan 2001
    */
    private HashMap entities = entities_init();

    // new since 14. Jan 2001
    private HashMap entities_init() {
    	HashMap map = new HashMap();
    	map.put("lt","<");
    	map.put("gt",">");
    	map.put("amp","&");
    	map.put("apos","\'");
    	map.put("quot","\"");
    	
    	return map;
    }
   
   
   /**
    * For debugging - lexer will print the generated tokens and the tokentypes to a Output. The default setting is 'false'.
    */
    private boolean debug = false; // for debugging, default setting is 'false'
    
    /**
     * For debugging - lexer will print the generated tokens and the tokentypes to a Output. The default setting is 'false'. 
     */
    public void setdebug(boolean b) {
    	this.debug = b;
    }
   
   
   /**
   *  Creating the Symbol that will be send to the parser.
   *  The corresponding Tokenname to the Tokentype can be
   *  found in sym.java. sym.java will be created by the parser.
   */
   private Symbol symbol(int type, Object value) {
	if (debug) {
		Output out = new Output();
		out.out("Lexer-Text: " +yytext() +" \t Tokentype: " +type+" \t Line = " +yyline, false);
	}
	return new Symbol(type, yyline, yycolumn, value);
   }


   /**
   *  Testing if the current yytext() containing a prefix that belong to the RDF namespace.
   */
   private boolean RDF_ns() {
	String Prefix = yytext();
	int i = Prefix.indexOf(":");
	if (i<0) Prefix = "";
	else Prefix = Prefix.substring(0,i);
	if (RDF_prefix_set.contains(Prefix)) {
//		System.out.println("NS found: "+Prefix);
		return true;
	}
	System.out.println("no RDF namespace Prefix: "+Prefix);
	return false;
   }

   /**
   *  Recognize the attribute "ID" and "xml..." and the special prefix "xmlns".
   */
   private Symbol nsattributename() {
	int i = yytext().indexOf(":");
	String attribute = yytext().substring(++i);
	String prefix = yytext().substring(0,--i);
//	System.out.println("NS-Prefix: "+prefix); 
	if (yytext().startsWith("xmlns"))  {
		ns = true;
		prefix = yytext().substring(++i);
		NS_Prefix = prefix;
		return symbol(sym.XMLNS, prefix);}
	if (yytext().startsWith("xml")) return symbol(sym.XMLATTR, yytext());
	if (attribute.equals("ID")) {
		yybegin(RDF_Attribute_IDSYMBOL);
		return symbol(sym.IDNAME, yytext());
	}
	return symbol(sym.ATTRIBUTENAME, yytext());
   }

   /**
   *  Recognize the attribute "ID" and "xml..." and the special prefix "xmlns".
   */
   private Symbol attributename() { 
	if (yytext().equals("xmlns")) {
		ns = true;
		NS_Prefix = "";
		return symbol(sym.XMLNS, "");
	}
	if (yytext().startsWith("xml")) return symbol(sym.XMLATTR, yytext());
	if (yytext().equals("ID"))  {
		yybegin(RDF_Attribute_IDSYMBOL);
		return symbol(sym.IDNAME, yytext());
	}
	return symbol(sym.ATTRIBUTENAME, yytext());
   }


   /**
   *  Replace a reference with the corresponding string.
   *  changed 14. Jan 2001
   */
   private String Ref_replace(String str) {
	int i = str.indexOf("&");
	if (i < 0) return str;
	String text = str.substring(0,i); // the text before the &
	String rest = str.substring(i); // the text starting with &
	// System.out.println("rest1: "+rest);
	int j = rest.indexOf(";");
	String value = rest.substring(0,j); // the text from & to the next ;
	rest = rest.substring(++j); // the rest string after the ;
	// System.out.println("rest2: "+rest);
	int k = 0;
	if (value.startsWith("&#x")) { // Hexadecimal
		value = value.substring(3);
		k = hextodec(value);
		char c = (char)k;
		text = text + c + Ref_replace(rest);
//		System.out.println("char Hex: "+c+" "+k+ " " +text);	
	} else {
		if (value.startsWith("&#")) { // Decimal
			value = value.substring(2);
			k = Integer.parseInt(value);
			char c = (char)k;
			text = text + c + Ref_replace(rest);
//			System.out.println("char "+c+" "+k+ " " +text);
		} else { // Entity References
			// remove the &
			value = value.substring(1);
			if (entities.containsKey(value)) {
				text = text + entities.get(value).toString() + Ref_replace(rest);			
			} else {
				System.out.println("Value "+value+" not found in HashMap.");
				text = text + Ref_replace(rest);
			}
		}
	}
	
	return text;
   }

   /**
   *  Transforming the hexadecimal number given by the parameter to the corresponding 
   *  decimal int.
   */
   private int hextodec(String str) {
	str = str.toLowerCase();
//	System.out.println(str);
	char[] ch_array = str.toCharArray();
	int i = ch_array.length;
	int j = 0;
	int mult = 1;
	int hex = 16;
	while (i-- > 0) {
//		System.out.println(ch_array[i]);
		if (ch_array[i] == 'f') j = j + (15 * mult);
		if (ch_array[i] == 'e') j = j + (14 * mult);
		if (ch_array[i] == 'd') j = j + (13 * mult);
		if (ch_array[i] == 'c') j = j + (12 * mult);
		if (ch_array[i] == 'b') j = j + (11 * mult);
		if (ch_array[i] == 'a') j = j + (10 * mult);
		if (ch_array[i] == '0' ||
		    ch_array[i] == '1' ||
		    ch_array[i] == '2' ||
		    ch_array[i] == '3' ||
		    ch_array[i] == '4' ||
		    ch_array[i] == '5' ||
		    ch_array[i] == '6' ||
		    ch_array[i] == '7' ||
		    ch_array[i] == '8' ||
		    ch_array[i] == '9' ) {
			String s = ch_array[i]+"";	
			int k = Integer.parseInt(s);
//			System.out.println(s+" als int "+k);
			j = j + (k * mult);
		}
		mult = mult * hex; 
	}
	System.out.println(str+" = "+j+" dec");
	return j;
   }

    // new since 14. Jan 2001
    // last change 01. Aug 2001
    private void setEntities(String text) {
   	//System.out.println("setEntities");
	String Name, Value;

   	int i = text.indexOf("ENTITY");
  	if (i < 0) return;
   	text = text.substring(i+6).trim(); // remove the "ENTITY" string
   	if (text.startsWith("%")) text = text.substring(2).trim();
   	i = text.indexOf(" ");
   	Name = text.substring(0,i).trim();
   	//System.out.println("Name: "+Name);
   	text = text.substring(i).trim();
 	if (text.startsWith("SYSTEM") || text.startsWith("PUBLIC")) {
 		System.out.println("The system does not support external entities!");  	
 		return;
 	}
 	if (! (text.startsWith("\"") || text.startsWith("'"))) {
 		System.out.println("ENTITY text should start with \" or '");
 		return;
 	} 
 	if (text.startsWith("\"")) {
 	 	text = text.substring(1);
	 	i = text.indexOf("\"");
 	} else {
 	 	text = text.substring(1);
		i = text.indexOf("'");	
 	}
	if (i < 0) {
		System.out.println("missing closing brackets in ENTITY declaration");
		return;
	}
 	Value = text.substring(0,i);
 	//System.out.println("Value: "+Value);
 	
 	// entering the Name, Value pair to the entiteis Hash Map
 	entities.put(Name, Value);
 	
 	// recursive call of the method
 	text = text.substring(i).trim();
 	setEntities(text);
   }



%}
// The numbers in brackets correspond to the numbers of 
// XML 1.0 (http://www.w3.org/TR/1998/REC-xml-19980210)
// [2] of XML not used
// Char 	= \u0009 | \u000A | \u000D | [\u0020-\uD7FF] | [\uE000-\uFFFD] | [\u10000-\u10FFFF]
// [3] of XML 
S		=  (\u0020 | \u009 | \u00D | \u00A)+
// [5] of XML
Name		= ({Letter} | "_" | ":") ({NameChar} | ":")* 
// [7] of XML
Nmtoken	= ({NameChar} | ":")+
// [9] of XML
EntityValue	= (\u0022 ( [^%&\"] | {PEReference} | {Reference})* \u0022) | (\u0027 ( [^%&'] | {PEReference} | {Reference})* \u0027) 
// [10] of XML
AttValue	= (\u0022 ( [^\"<&]| {Reference} )* \u0022) | (\u0027 ( [^'<&]| {Reference} )* \u0027)
// [11] of XML
SystemLiteral = (\u0022 [^\"]* \u0022) | (\u0027 [^']* \u0027)
// [12] of XML
PubidLiteral = (\u0022 ({PubidChar} | \u0027)* \u0022) | (\u0027 {PubidChar}* \u0027)
// [13] of XML without \u0027 (')
PubidChar	= \u0020 | \u000D | \u000A | [a-zA-Z0-9] | [-()+,./:=?;!*#@$_%]  
//Comments
// [15] of XML
Comment 	= "<!--" {CommentContend} "-->"
CommentContend 	= ([^-])* | (([^-])* "-" ([^-])+)*
// [16] of XML abbreviated
PI		= "<?" {String} "?>"
// [23] of XML
XMLDecl		= "<?xml" {VersionInfo} {EncodingDecl}? {SDDecl}? {S}? "?>" 
// [24] of XML
 VersionInfo 	= {S} "version" {Eq} ((\u0022 {VersionNum} \u0022) | (\u0027 {VersionNum} \u0027))
// [25] of XML
Eq		= {S}? "=" {S}?
// [26] of XML
 VersionNum 	= ([a-zA-Z0-9_.:] | "-")+
// [28] of XML
doctypedecl = "<!DOCTYPE" {S} {Name} ({S} {ExternalID})? {S}? ("[" ({markupdecl} | {PEReference} | {S})* "]" {S}?)? ">"
// [29] of XML
markupdecl = {elementdecl} | {AttlistDecl} | {EntityDecl} | {NotationDecl} | {PI} | {Comment}
// [32] of XML
SDDecl 		= {S} "standalone" {Eq} ((\u0027 ("yes" | "no") \u0027) | (\u0022 ("yes" | "no") \u0022))
// [33] of XML
LanguageID	= {Langcode} ( "-" {Subcode})*
// [34] of XML
Langcode	= {ISO639Code} | {IanaCode} | {UserCode}
// [35] of XML
ISO639Code	= ([a-zA-Z]) ([a-zA-Z])
// [36] of XML
IanaCode	= ("i" |"I") "-" ([a-zA-Z])+
// [37] of XML
UserCode	= ("x" | "X") "-" ([a-zA-Z])+
// [38] of XML
Subcode		= ([a-zA-Z])+
// [45] of XML
elementdecl	= "<!Element" {S} {Name} {S} {contentspec} {S}? ">"
// [46] of XML ??
contentspec = "EMPTY" | "ANY" | {Mixed} | {children}
// [47] of XML, modified to avoid nesting that causes an stack overflow
// original: children = ({choice} | {seq}) ("?" | "*" | "+")?
children	= [^>]*
// [48] of XML
//cp	= ({Name} | {choice} | {seq}) ("?" | "*" | "+")?
// [49] of XML
//choice	= "(" {S}? {cp} ({S}? "|" {S}? {cp})* {S}? ")"
// [50] of XML
//seq	= "(" {S}? {cp} ({S}? "," {S}? {cp})* {S}? ")"
// [51] of XML
Mixed	= "(" {S}? "#PCDATA" ({S}? "|" {S}? {Name})* {S}? ")*" | "(" {S}? "#PCDATA" {S}? ")"
// [52] of XML
AttlistDecl	= "<!ATTLIST" {S} {Name} {AttDef}* {S}? ">"
// [53] of XML
AttDef	= {S} {Name} {S} {AttType} {S} {DefaultDecl}
// [54] of XML
AttType	= {StringType} | {TokenizedType} | {EnumeratedType}
// [55] of XML
StringType	= "CDATA"
// [56] of XML
TokenizedType	= "ID" | "IDREF" | "IDREFS" | "ENTITY" | "ENTITIES" | "NMTOKEN" | "NMTOKENS"
// [57] of XML
EnumeratedType	= {NotationType} | {Enumeration}
// [58] of XML
NotationType	= "NOTATION" {S} "(" {S}? {Name} ({S}? "|" {S}? {Name})* {S}? ")"
// [59] of XML
Enumeration	= "(" {S}? {Nmtoken} ({S}? "|" {S}? {Nmtoken})* {S}? ")"
// [60] of XML
DefaultDecl	= "#REQUIRED" | "#IMPLIED" | (( "#FIXED" {S})? {AttValue})
// [66] of XML
CharRef	 	= "&#" [0-9]+ ";" | "&#x" [0-9a-fA-F]+ ";"
// [67] of XML
Reference 	= {EntityRef} | {CharRef}
// [68] of XML 
EntityRef 	= "&" {QName} ";"
// [69] of XML
PEReference	= "%" {Name} ";"
// [70] of XML
EntityDecl	= {GEDecl} | {PEDecl}
// [71] of XML
GEDecl	= "<!ENTITY" {S} {Name} {S} {EntityDef} {S}? ">"
// [72] of XML
PEDecl	= "<!ENTITY" {S} "%" {S} {Name} {S} {PEDef} {S}? ">"
// [73] of XML
EntityDef	= {EntityValue} | ({ExternalID} {NDataDecl}?)
// [74] of XML
PEDef	= {EntityValue} | {ExternalID} 
// [75] of XML
ExternalID	= "SYSTEM" {S} {SystemLiteral} | "PUBLIC" {S} {PubidLiteral} {S} {SystemLiteral}
// [76] of XML
NDataDecl	= {S} "NDATA" {S} {Name}
// [80] of XML
EncodingDecl 	= {S} "encoding" {Eq} (\u0027 {EncName} \u0027 | \u0022 {EncName} \u0022)
// [81] of XML
EncName		= [A-Za-z] ([A-Za-z0-9._] | "-")*
// [82] of XML
NotationDecl	= "<!NOTATION" {S} {Name} {S} (ExternalID} | {PublicID}) {S}? ">"
// [83] of XML
PublicID	= "PUBLIC" {S} {PubidLiteral}
//Characters
// [84] of XML
Letter 		= {BaseChar} | {Ideographic}
// [85] of XML
BaseChar 	= [\u0041-\u005A\u0061-\u007A\u00C0-\u00D6\u00D8-\u00F6\u00F8-\u00FF\u0100-\u0131\u0134-\u013E\u0141-\u0148\u014A-\u017E\u0180-\u01C3\u01CD-\u01F0\u01F4-\u01F5\u01FA-\u0217\u0250-\u02A8\u02BB-\u02C1\u0386\u0388-\u038A\u038C\u038E-\u03A1\u03A3-\u03CE\u03D0-\u03D6\u03DA\u03DC\u03DE\u03E0\u03E2-\u03F3\u0401-\u040C\u040E-\u044F\u0451-\u045C\u045E-\u0481\u0490-\u04C4\u04C7-\u04C8\u04CB-\u04CC\u04D0-\u04EB\u04EE-\u04F5\u04F8-\u04F9\u0531-\u0556\u0559\u0561-\u0586\u05D0-\u05EA\u05F0-\u05F2\u0621-\u063A\u0641-\u064A\u0671-\u06B7\u06BA-\u06BE\u06C0-\u06CE\u06D0-\u06D3\u06D5\u06E5-\u06E6\u0905-\u0939\u093D\u0958-\u0961\u0985-\u098C\u098F-\u0990\u0993-\u09A8\u09AA-\u09B0\u09B2\u09B6-\u09B9\u09DC-\u09DD\u09DF-\u09E1\u09F0-\u09F1\u0A05-\u0A0A\u0A0F-\u0A10\u0A13-\u0A28\u0A2A-\u0A30\u0A32-\u0A33\u0A35-\u0A36\u0A38-\u0A39\u0A59-\u0A5C\u0A5E\u0A72-\u0A74\u0A85-\u0A8B\u0A8D\u0A8F-\u0A91\u0A93-\u0AA8\u0AAA-\u0AB0\u0AB2-\u0AB3\u0AB5-\u0AB9\u0ABD\u0AE0\u0B05-\u0B0C\u0B0F-\u0B10\u0B13-\u0B28\u0B2A-\u0B30\u0B32-\u0B33\u0B36-\u0B39\u0B3D\u0B5C-\u0B5D\u0B5F-\u0B61\u0B85-\u0B8A\u0B8E-\u0B90\u0B92-\u0B95\u0B99-\u0B9A\u0B9C\u0B9E-\u0B9F\u0BA3-\u0BA4\u0BA8-\u0BAA\u0BAE-\u0BB5\u0BB7-\u0BB9\u0C05-\u0C0C\u0C0E-\u0C10\u0C12-\u0C28\u0C2A-\u0C33\u0C35-\u0C39\u0C60-\u0C61\u0C85-\u0C8C\u0C8E-\u0C90\u0C92-\u0CA8\u0CAA-\u0CB3\u0CB5-\u0CB9\u0CDE\u0CE0-\u0CE1\u0D05-\u0D0C\u0D0E-\u0D10\u0D12-\u0D28\u0D2A-\u0D39\u0D60-\u0D61\u0E01-\u0E2E\u0E30\u0E32-\u0E33\u0E40-\u0E45\u0E81-\u0E82\u0E84\u0E87-\u0E88\u0E8A\u0E8D\u0E94-\u0E97\u0E99-\u0E9F\u0EA1-\u0EA3\u0EA5\u0EA7\u0EAA-\u0EAB\u0EAD-\u0EAE\u0EB0\u0EB2-\u0EB3\u0EBD\u0EC0-\u0EC4\u0F40-\u0F47\u0F49-\u0F69\u10A0-\u10C5\u10D0-\u10F6\u1100\u1102-\u1103\u1105-\u1107\u1109\u110B-\u110C\u110E-\u1112\u113C\u113E\u1140\u114C\u114E\u1150\u1154-\u1155\u1159\u115F-\u1161\u1163\u1165\u1167\u1169\u116D-\u116E\u1172-\u1173\u1175\u119E\u11A8\u11AB\u11AE-\u11AF\u11B7-\u11B8\u11BA\u11BC-\u11C2\u11EB\u11F0\u11F9\u1E00-\u1E9B\u1EA0-\u1EF9\u1F00-\u1F15\u1F18-\u1F1D\u1F20-\u1F45\u1F48-\u1F4D\u1F50-\u1F57\u1F59\u1F5B\u1F5D\u1F5F-\u1F7D\u1F80-\u1FB4\u1FB6-\u1FBC\u1FBE\u1FC2-\u1FC4\u1FC6-\u1FCC\u1FD0-\u1FD3\u1FD6-\u1FDB\u1FE0-\u1FEC\u1FF2-\u1FF4\u1FF6-\u1FFC\u2126\u212A-\u212B\u212E\u2180-\u2182\u3041-\u3094\u30A1-\u30FA\u3105-\u312C\uAC00-\uD7A3]
// [86] of XML
Ideographic  	= [\u4E00-\u9FA5\u3007\u3021-\u3029] 
// [87] of XML
CombiningChar 	= [\u0300-\u0345\u0360-\u0361\u0483-\u0486\u0591-\u05A1\u05A3-\u05B9\u05BB-\u05BD\u05BF\u05C1-\u05C2\u05C4\u064B-\u0652\u0670\u06D6-\u06DC\u06DD-\u06DF\u06E0-\u06E4\u06E7-\u06E8\u06EA-\u06ED\u0901-\u0903\u093C\u093E-\u094C\u094D\u0951-\u0954\u0962-\u0963\u0981-\u0983\u09BC\u09BE\u09BF\u09C0-\u09C4\u09C7-\u09C8\u09CB-\u09CD\u09D7\u09E2-\u09E3\u0A02\u0A3C\u0A3E\u0A3F\u0A40-\u0A42\u0A47-\u0A48\u0A4B-\u0A4D\u0A70-\u0A71\u0A81-\u0A83\u0ABC\u0ABE-\u0AC5\u0AC7-\u0AC9\u0ACB-\u0ACD\u0B01-\u0B03\u0B3C\u0B3E-\u0B43\u0B47-\u0B48\u0B4B-\u0B4D\u0B56-\u0B57\u0B82-\u0B83\u0BBE-\u0BC2\u0BC6-\u0BC8\u0BCA-\u0BCD\u0BD7\u0C01-\u0C03\u0C3E-\u0C44\u0C46-\u0C48\u0C4A-\u0C4D\u0C55-\u0C56\u0C82-\u0C83\u0CBE-\u0CC4\u0CC6-\u0CC8\u0CCA-\u0CCD\u0CD5-\u0CD6\u0D02-\u0D03\u0D3E-\u0D43\u0D46-\u0D48\u0D4A-\u0D4D\u0D57\u0E31\u0E34-\u0E3A\u0E47-\u0E4E\u0EB1\u0EB4-\u0EB9\u0EBB-\u0EBC\u0EC8-\u0ECD\u0F18-\u0F19\u0F35\u0F37\u0F39\u0F3E\u0F3F\u0F71-\u0F84\u0F86-\u0F8B\u0F90-\u0F95\u0F97\u0F99-\u0FAD\u0FB1-\u0FB7\u0FB9\u20D0-\u20DC\u20E1\u302A-\u302F\u3099\u309A] 
// [88] of XML
Digit 		=  [\u0030-\u0039\u0660-\u0669\u06F0-\u06F9\u0966-\u096F\u09E6-\u09EF\u0A66-\u0A6F\u0AE6-\u0AEF\u0B66-\u0B6F\u0BE7-\u0BEF\u0C66-\u0C6F\u0CE6-\u0CEF\u0D66-\u0D6F\u0E50-\u0E59\u0ED0-\u0ED9\u0F20-\u0F29] 
// [89] of XML
Extender 	=    [\u00B7\u02D0\u02D1\u0387\u0640\u0E46\u0EC6\u3005\u3031-\u3035\u309D-\u309E\u30FC-\u30FE] 




RDFBegin	= "RDF" | {QName} ":RDF"
RDFEnd		= "RDF" | {QName} ":RDF"
ResourceAttr	= "resource" | {QName} ":resource"
bagID		= "bagID" | {QName} ":bagID"
IDSymbol	= (\u0022 {QName} \u0022) | (\u0027 {QName} \u0027)
Resource	= (\u0022 "Resource" \u0022) | (\u0027 "Resource" \u0027)
Literal		= (\u0022 | \u0027) "Literal" (\u0022 | \u0027)
Collection		= (\u0022 | \u0027) "Collection" (\u0022 | \u0027)
Attributename	= ({Letter} | "_" ) ({NameChar})*
NSAttributename = {Attributename} ":" {Attributename} 
XMLlang		= "xml:lang" {Eq} ((\u0022 {LanguageID} \u0022) | (\u0027 {LanguageID} \u0027))
ParseTypeLit 	= "parseType" {Eq} {Literal} | {QName} ":parseType" {Eq} {Literal}
ParseTypeRes 	= "parseType" {Eq} {Resource} | {QName} ":parseType" {Eq} {Resource}
ParseTypeCol	= "parseType" {Eq} {Collection} | {QName} ":parseType" {Eq} {Collection}
String		= ( [^<>&] | {Reference} )*

ELSTART		= "<"
ELEND		= "</"
ALEND		= ">"
ALELEND		= "/>"

//XML
XMLDTD		= {doctypedecl} | {markupdecl}
HTML		= "<" {HTML_Name} ({S} {String})? ">" | "<" {HTML_Name} {S}? "/>" | "</" {HTML_Optional} {S}? ">"
HTML_Name	= {HTML_Forbidden} | {HTML_Optional}
// The lists for 'HTML_Forbidden' and 'HTML_Optional' are created from the 'Index of Elements' 
// of the HTML 4.0 Standard. 
// The 'Index of Elements' can be found at: http://www.w3.org/TR/REC-html40/index/elements.html.
// HTML-tags with forbidden end tags: 
HTML_Forbidden	= "AREA" | "area" | "BASE" | "base" | "BASEFONT" | "basefont" | "BR" | "br" | "COL" | "col" | "FRAME" | "frame" | "HR" | "hr" | "IMG" | "img" | "INPUT" | "input" | "ISINDEX" | "isindex" | "LINK" | "link" | "META" | "meta" | "PARAM" | "param" 
// HTML-tags with optional end tags (HTML-tag escaped):
HTML_Optional	= "BODY" | "body" | "COLGROUP" | "colgroup" | "DD" | "dd" | "DT" | "dt" | "HEAD" | "head" | "LI" | "li" | "OPTION" | "option" | "P" | "p" | "TBODY" | "tbody" | "TD" | "td" | "TFOOT" | "tfoot" | "THEAD" | "thead" | "TH" | "th" | "TR" | "tr"   

// From Namespaces in XML at http://www.w3.org/TR/1999/REC-xml-names-19990114/.
// [5] of Namespaces in XML
NameChar 	=   {Letter} | {Digit} | "." | "-" | "_" | {CombiningChar} | {Extender} 
// [4] of Namespaces in XML
QName 	 	=  ({Letter} | "_" ) ({NameChar})* 
// like [6] of Namespaces in XML
NSQName		= {QName} ":" {QName}


// The states of the lexer. 
/**
 * The state the lexer will start his work.
 */
%state YYINITIAL 

/** 
 * A state of the lexer.
 */
%state QName_Begin

/** 
 * A state of the lexer.
 */
%state QName_End

/** 
 * A state of the lexer.
 */
%state Attribute

// The states during a RDF description.
/** 
 * A state of the lexer.
 */
%state RDF_YYINITIAL

/** 
 * A state of the lexer.
 */
%state RDF_QName_Begin

/** 
 * A state of the lexer.
 */
%state RDF_QName_End

/** 
 * A state of the lexer.
 */
%state RDF_Attribute

/** 
 * A state of the lexer.
 */
%state RDF_Attribute_IDSYMBOL

%%

// Part three: lexical rules and actions

<YYINITIAL> {
   // comments
   {XMLDecl}		{ return symbol(sym.XMLSPEC, yytext()); }
   {Comment}		{ /* ignore */ 
			  /* System.out.println("Comment"); */ }

   // whitespace
   {S}			{ /* ignore */ 
			  /* System.out.println("S"); */ }
  {XMLDTD} 		{ /* set ENTITIES if exist */
   				setEntities(yytext()); }
   {HTML}		{ /* ignore */ 
			  /* System.out.println("HTML: "+yytext()); */ }

   // keywords
   {ELSTART}		{ yybegin(QName_Begin);
			  return symbol(sym.ELSTART, yytext()); }
   {ELEND}		{ yybegin(QName_End);
			  return symbol(sym.ELEND, yytext()); }
   {ALEND}		{ return symbol(sym.ALEND, yytext()); }
   {String}		{ /* ignore */ }

} // end YYINITIAL

<RDF_YYINITIAL> {
   // comments
   {Comment}		{ /* ignore */ 
			  /* System.out.println("Comment"); */ }

   // whitespace
   {S}			{ /* ignore */ 
			  /* System.out.println("S"); */ }

   // keywords
   {ELSTART}		{ yybegin(RDF_QName_Begin);
			  return symbol(sym.ELSTART, yytext()); }
   {ELEND}		{ yybegin(RDF_QName_End);
			  return symbol(sym.ELEND, yytext()); }
   {ALEND}		{ return symbol(sym.ALEND, yytext()); }
   {String}		{ String text = Ref_replace(yytext());
			  return symbol(sym.STRING, text); }

} // end RDF_YYINITIAL

<QName_Begin> {
   {RDFBegin}		{ yybegin(RDF_Attribute);
			  return symbol(sym.RDFBEGIN, yytext()); }
   {QName}		{ if (yytext().equals("Description")) {
				yybegin(RDF_Attribute);
				  return symbol(sym.QNAME, yytext());
			  } else {
				yybegin(Attribute);
				return symbol(sym.XMLQNAME, yytext().toUpperCase());
			  }
 			}
   {NSQName}		{ if (yytext().endsWith("Description")) {
				yybegin(RDF_Attribute);
			  	return symbol(sym.QNAME, yytext());
			  } else {
				yybegin(Attribute);
				return symbol(sym.XMLQNAME, yytext().toUpperCase());
			  }
 			}
} // end QName_Begin

<RDF_QName_Begin> {
   {QName}		{ yybegin(RDF_Attribute);
			  if (yytext().equals("Description")) nested_counter++;
			  return symbol(sym.QNAME, yytext()); }
   {NSQName}		{ yybegin(RDF_Attribute);
			  if (yytext().endsWith("Description")) nested_counter++;
			  return symbol(sym.QNAME, yytext()); }
} // end RDF_QName_Begin


<QName_End> {
   {QName}		{ yybegin(YYINITIAL);
			  return symbol(sym.XMLQNAME, yytext().toUpperCase());
			}
   {NSQName}		{ yybegin(YYINITIAL);
			  return symbol(sym.XMLQNAME, yytext().toUpperCase());
			}
} // end QName_End

<RDF_QName_End> {
  {RDFEnd}		{ yybegin(YYINITIAL);
			  return symbol(sym.RDFEND, yytext()); }
   {QName}		{ if (yytext().equals("Description")) {
				if (nested_counter == 0) yybegin(YYINITIAL); 
				  else {
					yybegin(RDF_YYINITIAL);
					nested_counter--;
				  }
			  } else yybegin(RDF_YYINITIAL);
			  return symbol(sym.QNAME, yytext()); }
   {NSQName}		{ if (yytext().endsWith("Description")) {
				if (nested_counter == 0) yybegin(YYINITIAL); 
				  else {
					yybegin(RDF_YYINITIAL);
					nested_counter--;
				  }
			  } else yybegin(RDF_YYINITIAL);
			  return symbol(sym.QNAME, yytext()); }

} // end RDF_QName_End

<Attribute> {

   // whitespace
   {S}			{ /* ignore */ 
			  /* System.out.println("S"); */ }
   {XMLlang}		{ /* ignore */ 
			  /* System.out.println("XMLlang"); */}
   {NSAttributename}	{ if (yytext().startsWith("xmlns")) {
				ns = true;
				int i = yytext().indexOf(":");
				String prefix = yytext().substring(++i);
				NS_Prefix = prefix;
				return symbol(sym.XMLNS, prefix);
			  }
			}
   {Attributename}	{ if (yytext().startsWith("xmlns")) {
				ns = true;
				String prefix = "";
				NS_Prefix = prefix;
				return symbol(sym.XMLNS, prefix);
			  }
			}
   {Eq}			{ if (ns) return symbol(sym.EQUAL, yytext()); }
   {AttValue}		{ if (ns) {
				  if (yytext().equals("\""+rdf.NS+"\"")) RDF_prefix_set.add(NS_Prefix);
				  String text = Ref_replace(yytext());
				  ns = false;
				  return symbol(sym.ATTRIBUTEVALUE, text);
			  }
			}
   {ALEND}		{ yybegin(YYINITIAL); 
			  return symbol(sym.ALEND, yytext()); }
   {ALELEND}		{ yybegin(YYINITIAL); 
			  return symbol(sym.ALELEND, yytext()); }
} // end Attribute

<RDF_Attribute> {

   // whitespace
   {S}			{ /* ignore */ 
			  /* System.out.println("S"); */ }
   {ResourceAttr}	{ return symbol(sym.RESOURCEATTR, yytext()); }
   {bagID}		{ yybegin(RDF_Attribute_IDSYMBOL);
			  return symbol(sym.BAGID, yytext()); }
   {ParseTypeLit}	{ return symbol(sym.PARSETYPELIT, yytext()); }
   {ParseTypeRes}	{ return symbol(sym.PARSETYPERES, yytext()); }
   {ParseTypeCol} { return symbol(sym.PARSETYPECOL, yytext()); }
   {XMLlang}		{ return symbol(sym.XMLLANG, yytext()); }

   {NSAttributename}	{ return nsattributename(); }
   {Attributename}	{ return attributename(); }
   {ALEND}		{ yybegin(RDF_YYINITIAL); 
			  return symbol(sym.ALEND, yytext()); }
   {ALELEND}		{ yybegin(RDF_YYINITIAL); 
			  return symbol(sym.ALELEND, yytext()); }
   {Eq}			{ return symbol(sym.EQUAL, yytext()); }
   {AttValue}		{ if (ns) {
				if (yytext().equals("\""+rdf.NS+"\"")) RDF_prefix_set.add(NS_Prefix);
				ns = false;
			  }
			  String text = Ref_replace(yytext());
			  return symbol(sym.ATTRIBUTEVALUE, text); }

} // end RDF_Attribute

<RDF_Attribute_IDSYMBOL> {
   {Eq}			{ return symbol(sym.EQUAL, yytext()); }
   {IDSymbol}		{ yybegin(RDF_Attribute);
			  return symbol(sym.ATTRIBUTEVALUE, yytext()); }

} // end RDF_Attribute_IDSYMBOL


/* error fallback */
.|\n			{ String state="yystate";
			  switch (yystate()) {
				case 0: state = "YYINITIAL"; break;
				case 4: state = "RDF_YYINITIAL"; break;
				case 1: state = "QName_Begin"; break;
				case 5: state = "RDF_QName_Begin"; break;
				case 2: state = "QName_End"; break;
				case 6: state = "RDF_QName_End"; break;
				case 3: state = "Attribute"; break;
				case 7: state = "RDF_Attribute"; break;
				case 8: state = "RDF_Attribute_IDSYMBOL"; break;
			  }	
			  RDF_Error e = new RDF_Error();
			  e.emit_error(0, yystate(), "<"+yytext()+"> at line "+yyline+", column "+yycolumn+", state "+state+".", "lexer:error fallback");
			  //changed on 14. Jan. 2001: 
			  // System.esit(0);
			  yy_do_eof();
			  // throw new Error("<"+yytext()+"> at line "+yyline+", column "+yycolumn+", state "+state+".");
			 }
