package eworks.RQL.model;

/**
 * <p>Represents a list of <code>Projection</code>-instances, that is the list
 * of variables that are part of a query's result (other variables may be bound
 * as well, but will not be returned as the query's result). This aspect and
 * terminology is copied from database query languages like SQL.</p>
 * 
 * @author Fabian Wleklinski (<a href="mailto:fabian@wleklinski.de">fabian@wleklinski.de</a>)
 * @version 1.00 (2003-11-03)
 * @see Projection
 * @see <a href="http://139.91.183.30:9090/RDF/RQL/" target="_blank">
 *      The RDF Query Language (RQL)</a>
 */
public class ProjectionList {

	private Projection[] projections;

	/**
	 * Creates a new and empty projection list.
	 */
	public ProjectionList() {
		super();
		this.projections = new Projection[0];
	}

	/**
	 * Creates a new projection list and encapsulates a certain projection.
	 * 
	 * @param projection The projection list's only encapsulated projection.
	 */
	public ProjectionList( Projection projection ) {
		super();
		this.projections = new Projection[] { projection };
	}

	/**
	 * Returns the number of projections encapsulated by this projection list.
	 * 
	 * @return The number of projections.
	 */
	public int getLength() {
		return this.projections.length;
	}
	
	/**
	 * Returns the i-th projection that is encapsulated by this projection list.
	 * 
	 * @param i The number of the projection to return.
	 * @return The i-th projection (0 <= i <= <code>getLength()</code>).
	 * @throws ArrayIndexOutOfBoundsException if <code>i</code> is negative or not
	 *         lower than <code>getLength()</code>.
	 * @see #getLength()
	 */
	public Projection get(int i) {
		return this.projections[i];
	}
	
	/**
	 * Adds a new projection to this projection list.
	 * 
	 * @param projection The projection to add.
	 */
	public void add(Projection projection) {
		Projection[] projections = new Projection[ this.projections.length + 1 ];
				
		System.arraycopy(this.projections,0,projections,0,this.projections.length);
		projections[this.projections.length] = projection;
		this.projections = projections;
	}

	/**
	 * Returns this projection list's string representation.
	 * 
	 * @return This projection list's string representation.
	 */
	public String toString() {
		String result = " ProjectionList[";
		for (int i=0; i<this.projections.length; i++) {
			if (i>0)
				result += ",";
			result += this.projections[i].toString();
		}
		result += "]";
			 
		return result;
	}
}
