package eworks.RQL.model;

import eworks.RQL.engine.VariableNotBoundException;

/**
 * <p>A condition that represents a disjunction of several other conditions,
 * so called &quot;subconditions&quot;. (A disjunction is expressed using the
 * <code>OR</code>-operator according to the RQL syntax.)</p>
 * 
 * @author Fabian Wleklinski (<a href="mailto:fabian@wleklinski.de">fabian@wleklinski.de</a>)
 * @version 1.00 (2003-11-03)
 * @see <a href="http://139.91.183.30:9090/RDF/RQL/" target="_blank">
 *      The RDF Query Language (RQL)</a>
 */
public class DisjunctionCondition extends CompositionCondition {

	/**
	 * Creates an empty disjunction condition (with no subconditions). 
	 */
	public DisjunctionCondition() {
		super();
	}
	
	/**
	 * Creates a disjunction condition using certain subconditions.
	 * 
	 * @param subConditions This condition's subconditions.
	 */	
	public DisjunctionCondition( Condition[] subConditions ) {
		super(subConditions);
	}

	/**
	 * Creates a disjunction condition using one certain subcondition.
	 * 
	 * @param subCondition This condition's only subcondition.
	 */	
	public DisjunctionCondition( Condition subCondition ) {
		super(subCondition);
	}

	/**
	 * <p>Checks if this disjunction condition is satisfied using a certain symbol
	 * table. Therefore it is checked for all subconditions if they are satisfied
	 * (using the same symbol table). The result is <code>false</code> if and only if
	 * no subcondition matches.</p>
	 * <p>This method uses so called &quot;short circuit evaluation&quot;, that
	 * means the <code>true</code> is returned immediately after the first
	 * subcondition is found that does match, and all further
	 * subconditions are not evaluated any more.</p>
	 * 
	 * @param symbols The symbol table to use.
	 * @return <code>true</code> if this disjunction condition is satisfied,
	 *         <code>false</code> else.
	 * @throws VariableNotBoundException if one or several needed values of this
	 *         condition's values or it's subcondition's values are not bound by
	 *         the given symbol table.
	 */
	public boolean matches(eworks.RQL.engine.SymbolTable symbols) throws VariableNotBoundException {
		for (int i=this.subConditions.length-1; i>=0; i--) {
			if (this.subConditions[i].matches(symbols))
				return true;
		}
		return false;
	}
}
