package eworks.RQL.model;

import eworks.RDF.model.*;
import eworks.RQL.engine.VariableNotBoundException;

/**
 * <p>Represents a condition that compares two values against each other.</p>
 * 
 * @author Fabian Wleklinski (<a href="mailto:fabian@wleklinski.de">fabian@wleklinski.de</a>)
 * @version 1.00 (2003-11-03)
 * @see <a href="http://139.91.183.30:9090/RDF/RQL/" target="_blank">
 *      The RDF Query Language (RQL)</a>
 */
public class ComparisonCondition extends Condition {

	/**
	 * The first value to compare.
	 */
	protected Value op1;

	/**
	 * The second value to compare.
	 */
	protected Value op2;

	/**
	 * The operator used to compare both values against each other.
	 */
	protected CompareOperator op;

	/**
	 * Creates a new comparison condition using two certain values and
	 * a certain operator.
	 * 
	 * @param op1 The first value to compare.
	 * @param op The operator used to compare both values against each other.
	 * @param op2 The second value to compare.
	 */
	public ComparisonCondition( Value op1, CompareOperator op, Value op2 ) {
		this.op1 = op1;
		this.op2 = op2;
		this.op = op;
	}
	
	/**
	 * Checks if this condition is satisfied by both values, using a
	 * certain symbol table.
	 * 
	 * @param symbols The symbol table to use.
	 * @return <code>true</code> if this condition is satisfied,
	 *         <code>false</code> else.
	 * @throws VariableNotBoundException if one or both of this condition's
	 *         values are not bound by the given symbol table.
	 */
	public boolean matches(eworks.RQL.engine.SymbolTable symbols)
	throws VariableNotBoundException {
		return this.op.compare(this.op1.getValue(symbols),this.op2.getValue(symbols));
	}
	
	/**
	 * Returns this condition's string representation.
	 * 
	 * @return This condition's string representation.
	 */
	public String toString() {
			return " " + super.toString() + "[" + this.op1 + " " + this.op + " " + this.op2 + "]";
	}
}