package eworks.RDF.model;

import java.util.Collection;

/**
 * <p>Represents the interface of a set of tuples. Therefore this interface provided
 * methods to manage instances that implement the {@link Tuple} interface.</p>
 * 
 * <p>Managed tuples may be grouped to tuple groups, therefore methods like
 * {@link #getNumberOfGroups} and {@link #getTuplesByGroup} exist.</p>
 * 
 * <p>This class allows duplicate tuples.</p>
 * 
 * @author Fabian Wleklinski (<a href="mailto:fabian@wleklinski.de">fabian@wleklinski.de</a>)
 * @version 1.00 (2003-11-03)
 */
public interface Tuples extends Collection {
	
	/**
	 * Adds the {@link Tuple} <code>t</code>. The tuples is added even if it
	 * is already contained, so it can be contained multiple times.
	 * 
	 * @param t The tuple to add.
	 * @return <code>true</code> if and only if the set of tuples has changed.
	 */
	public abstract boolean add(Tuple t);
	
	/**
	 * Returns the number of tuple groups. If there are no explicit tuple
	 * groups defined, derived classes should return the value 1.
	 * 
	 * @return The number of tuple groups.
	 */
	public abstract int getNumberOfGroups();
	
	/**
	 * Returns all tuples of a certain tuple groups, that is specified by it's
	 * numerical index. The returned tuples are returned using a class that implements
	 * the {@link Tuples} interface as well.
	 * 
	 * @param groupIndex The group's index.
	 * @return The tuples of the tuple group <code>index</code>.
	 * @throws ArrayIndexOutOfBoundsException if this tuple group does not
	 *         exist, see {@link #getNumberOfGroups()}.
	 */
	public abstract Tuples getTuplesByGroup(int groupIndex);
	
	/**
	 * Adds all tuples that are contained inside the given collection <code>c</code>.
	 * 
	 * @param c The tuples to add.
	 * @param distinct If set to <code>true</code>, there will no duplicates be added.
	 * @return <code>true</code> if and only if at least one tuple was added.
	 */
	public abstract boolean addAll(Collection c, boolean distinct);
	
	/**
	 * Returns a certain tuple specified by it's numerical index.
	 * 
	 * @param index The numerical index of the tuple to return.
	 * @return The tuple with the given <code>index</code>.
	 * @throws ArrayIndexOutOfBoundsException if this tuple does not exist,
	 *         see {@link #size()}.
	 */
	public abstract Tuple getTuple(int index);
	
	/**
	 * Clones this instance by creating a flat copy. (So the encapsulated tuples
	 * are not cloned as well.)
	 * 
	 * @return A flat copy of this instance implementing the <code>Tuples</code>
	 *         interface again.
	 */
	public Object clone();
}